// NTSC TV Signal Emulation
in vec2 TexCoord;
out vec4 FragColor;
uniform sampler2D InputTexture;
uniform float timer;

// How much R, G and B are offset : default is -0.333 pixels in fake-pixel-space
const float ColorFringeIntensity = -0.333;
// How much luminosity is output by a fake-pixel
const float FakePixelMaskGain = 0.75f;
// How much luminosity is output between fake-pixels (adds to the fake-pixel value)
const float FakePixelMaskOffset = 0.55f;
// How sharp will appear the pixels (Horizontal Sharpness, Vertical Sharpness A.K.A Scanlines)
const vec2 FakePixelMaskPower = vec2(0.0f, 0.0f);
// Scanline Off Sync (Slides one line out of two)
const float ScanlineOffSync = 0.25;
// Base Brightness
const float BaseBrightness = 0.8f;

// How much the Fake-Pixel effect is Active (0.0 = normal image, 1.0 = full FakePixel Effect)
const float FakePixelEffectBlend = 0.7f;

// Ghost Sampling : enable define to activate
const float GhostLatencyIntensity = 0.02f;
// Number of samples (higer is slower)
const int GhostNumSamples = 16;
// Latency of the RGB Signal (per-signal, in screen width percentage)
const vec3 SignalLatencyRGB = vec3(0.184f, 0.08f, 0.0624f);
// Attenuation of the ghosting latency
const float SignalLatencyAttenuation = 1.0f;

////////////////////////////////////////////////////////////////////
float saturate(float x) { return max(0.0, min(1.0, x)); }
vec2 saturate(vec2 x) { return max(vec2(0.0, 0.0), min(vec2(1.0, 1.0), x)); }
float expow(float value, float exponent) { return mix(1.0f, pow(value, max(exponent, 1.0f)), saturate(exponent)); }

// the code that calls expow() carefully builds vec2 for some reason and calls this only to have it implicitly thrown away (which is a warning)
// so this was added to get rid of the warning
float expow(vec2 value, vec2 exponent) { return mix(1.0f, pow(value.x, max(exponent.x, 1.0f)), saturate(exponent.x)); }
float expow(vec2 value, float exponent) { return mix(1.0f, pow(value.x, max(exponent, 1.0f)), saturate(exponent)); }

// MultiSampling for ghosting effect
vec3 GhostSample(vec2 t, float latency) {
	vec3 Out = texture(InputTexture, t).rgb;
	float Weight = 1.0f;
	vec2 Direction = vec2(-latency, 0.0f);
	for(int i = 1; i < GhostNumSamples; i++) {
		float curweight = pow(1.0f - (float(i) / GhostNumSamples), 1.0f / SignalLatencyAttenuation);

		Out += GhostLatencyIntensity * curweight * texture(InputTexture, saturate(t + (1.0f - curweight) * Direction)).xyz;
		Weight += GhostLatencyIntensity * curweight;
	}
	return Out / Weight;
}

// Compositing of the TV Emulation
void main() {
	// Apply Off-Sync
	vec2 IResolution = textureSize(InputTexture, 0);
	vec2 IPosition = TexCoord + (ScanlineOffSync / IResolution.x) * vec2(sin(timer + (TexCoord.y * 3.1415926 * IResolution.y)), 0);

	// Sampling 3 Images biased to simulate TV RGB Offset
	vec3 LatencyWeight = vec3(0.0f, 0.0f, 0.0f);
	for(int i = 1; i < GhostNumSamples; i++) { LatencyWeight += texture(InputTexture, IPosition + vec2(1.0f / IResolution.x, 0.0f)).xyz; }
	vec3 LatencyRGB = SignalLatencyRGB * (1.0 - (LatencyWeight / GhostNumSamples));

	// Apply base Brightness
	vec3 SMP_Value = vec3(
		GhostSample(IPosition, LatencyRGB.x).x,
		GhostSample(IPosition + ((vec2(ColorFringeIntensity, 0.0f)) / IResolution), LatencyRGB.y).y,
		GhostSample(IPosition + ((vec2(ColorFringeIntensity * 2.0f, 0.0f)) / IResolution), LatencyRGB.z).z
	 ) * BaseBrightness;

	// CRT Cell Masks (HorizontalRGB+Scanline)
	float PixelMaskScanline = pow(saturate(4 * fract(IPosition.y * IResolution.y) * (1.0f - fract(IPosition.y * IResolution.y))), FakePixelMaskPower.y);
	vec3 PixelRGB = ((vec3(
		expow(saturate(4 * fract(IPosition.x * IResolution.x) * (1.0f - fract(IPosition.x * IResolution.x))), FakePixelMaskPower.x),
		expow(saturate(4 * fract(IPosition.x * IResolution.x + vec2(ColorFringeIntensity, 0.0f)) * (1.0f - fract(IPosition.x * IResolution.x + vec2(ColorFringeIntensity, 0.0f)))), FakePixelMaskPower.x),
		expow(saturate(4 * fract(IPosition.x * IResolution.x + vec2(ColorFringeIntensity * 2.0f, 0.0f)) * (1.0f - fract(IPosition.x * IResolution.x + vec2(ColorFringeIntensity * 2.0f, 0.0f)))), FakePixelMaskPower.x)
	) * PixelMaskScanline * FakePixelMaskGain) + FakePixelMaskOffset) * SMP_Value;

	// Non-Pixelated Image
	vec4 OriginalRGB = texture(InputTexture, IPosition) * 256.0;
	int DitherV = (int(gl_FragCoord.x + gl_FragCoord.y) & 1) << 2;
	vec3 ImageRGB = vec3(int(OriginalRGB.x + DitherV) >> 3, int(OriginalRGB.y + DitherV) >> 3, int(OriginalRGB.z + DitherV) >> 3) / 32.0;

	//Output
	FragColor = vec4(mix(ImageRGB, PixelRGB, FakePixelEffectBlend), 1.0);
}