in vec4 pixelpos;
in vec4 vColor;
in vec3 vWorldNormal;
in vec3 vEyeNormal;
#ifdef SHADER_PASS_LIGHTS
in vec3 vLightVertex;
#endif
in vec2 vTexCoord;

out vec4 FragColor;
#ifdef SHADER_PASS_GBUFFER
out vec4 FragNormal;
#endif

struct Material {
	vec4 Base;
	vec4 Bright;
	vec3 Normal;
	vec3 Specular;
	float Glossiness;
	float SpecularLevel;
	float Metallic;
	float Roughness;
	float AO;
};

vec4 Process(vec4 color);
vec4 ProcessTexel();
Material ProcessMaterial();
vec3 MaterialInitialLightValue(Material mat, vec3 color);
vec3 ProcessMaterialLight(Material material, vec3 color);

// [Sherbet] Screen Space Reflections
#ifdef SHADER_PASS_SSR
#define SSR_RAYSTEP 64.f
#define SSR_ITERATIONCOUNT 80
#define SSR_DISTANCEBIAS 2.f

vec3 SSRgetPosFromDepth(vec2 texturePos, float depth) {
	vec4 inversed = inverse(ProjectionMatrix) * vec4(texturePos, depth, 1.f); // going back from projected
	return inversed.xyz / inversed.w;
}

vec4 SSRgenProjPos(vec3 pos){
	vec4 samplePosition = ProjectionMatrix * vec4(pos, 1.f);
	samplePosition.xy /= samplePosition.w;
	return vec4(samplePosition.xy * 0.5 + 0.5, samplePosition.xy);
}

vec3 SSRsampleCol(vec2 screenPosition) {
	if(screenPosition.x < 0.0 || screenPosition.x > 1.0 || screenPosition.y < 0.0 || screenPosition.y > 1.0) { return vec3(0.0); }
	float fragDist = distance(screenPosition, vec2(0.5)) * 2.0;
	return texture(SSR_COLOR, screenPosition).rgb * max(0.0, 1.0 - fragDist);
}

vec3 SSR(vec3 pos, vec3 eyeNormal) {
	vec4 clipPos = ProjectionMatrix * ViewMatrix * vec4(pos, 1.0); clipPos.xy /= clipPos.w; // Perspective divide
	vec3 marchingPosition = SSRgetPosFromDepth(clipPos.xy, texture(SHADER_PASS_SSR, clipPos.xy * 0.5 + 0.5).x);
	vec3 step = SSR_RAYSTEP * normalize(reflect(marchingPosition, normalize(eyeNormal)));
	marchingPosition += step;
	// Standard search
	float delta;
	for(int i = 0; i < SSR_ITERATIONCOUNT; i++) {
		// Check hit
		vec4 screenPosition = SSRgenProjPos(marchingPosition);
		if(abs(screenPosition.w) > 1 || abs(screenPosition.z) > 1) { return vec3(0.0); }
		float depthFromScreen = abs(SSRgetPosFromDepth(screenPosition.zw, texture(SHADER_PASS_SSR, screenPosition.xy).x).z);
		delta = abs(marchingPosition.z) - depthFromScreen;
		if(abs(delta) < SSR_DISTANCEBIAS) { return SSRsampleCol(screenPosition.xy); }
		if(delta > 0) { break; }
		// This is sort of adapting step, should prevent lining reflection by doing sort of iterative converging
		// Some implementation doing it by binary search, but I found this idea more cheaty and way easier to implement
		float directionSign = sign(delta);
		step = step * (1.0 - SSR_RAYSTEP * max(directionSign, 0.0));
		marchingPosition -= step * directionSign;
	}
	// Perform binary search
	for(int i = 0; i < SSR_ITERATIONCOUNT; i++) {
		step *= 0.5;
		marchingPosition -= step * sign(delta);
		// Check hit
		vec4 screenPosition = SSRgenProjPos(marchingPosition);
		if(abs(screenPosition.w) > 1 || abs(screenPosition.z) > 1) { return vec3(0.0); }
		float depthFromScreen = abs(SSRgetPosFromDepth(screenPosition.zw, texture(SHADER_PASS_SSR, screenPosition.xy).x).z);
		delta = abs(marchingPosition.z) - depthFromScreen;
		if(abs(delta) < SSR_DISTANCEBIAS) { return SSRsampleCol(screenPosition.xy); }
	}
	return vec3(0.0);
}
#endif

// Desaturate a color
vec4 desaturate(vec4 texel) {
	if(uDesaturationFactor > 0.0) {
		float gray = (texel.r * 0.3 + texel.g * 0.56 + texel.b * 0.14);
		return mix(texel, vec4(gray, gray, gray, texel.a), uDesaturationFactor);
	}
	return texel;
}

// The same but for RGB instead of RGBA
vec3 desaturateRGB(vec3 texel) {
	if(uDesaturationFactor > 0.0) { float gray = (texel.r * 0.3 + texel.g * 0.56 + texel.b * 0.14); return mix(texel, vec3(gray), uDesaturationFactor); }
	return texel;
}

// This function is common for all (non-special-effect) fragment shaders
vec4 getTexel(vec2 st) {
	vec4 texel = texture(texture1, st);

	// Apply texture modes
	switch(uTextureMode) {
	case 1: // TM_MASK
		texel.rgb = vec3(1.0, 1.0, 1.0);
		break;
	case 2: // TM_OPAQUE
		texel.a = 1.0;
		break;
	case 3: // TM_INVERSE
		texel.rgb = 1.0 - texel.rgb;
		break;
	case 4: // TM_REDTOALPHA
		float gray = (texel.r * 0.3 + texel.g * 0.56 + texel.b * 0.14);
		texel = vec4(1.0, 1.0, 1.0, gray * texel.a);
		break;
#ifdef SHADER_PASS_SSR
	case 5: // TM_SSR
		texel = vec4(mix(texel.rgb, SSR(pixelpos.xyz, vEyeNormal), uAddColor.a), 1.0);
		break;
#endif
	}

	texel.rgb += uAddColor.rgb;
	if(uObjectColor2.a == 0.0) {
		texel *= uObjectColor;
	} else {
		float topatpoint = (uGradientTopPlane.w + uGradientTopPlane.x * pixelpos.x + uGradientTopPlane.y * pixelpos.z) * uGradientTopPlane.z;
		float bottomatpoint = (uGradientBottomPlane.w + uGradientBottomPlane.x * pixelpos.x + uGradientBottomPlane.y * pixelpos.z) * uGradientBottomPlane.z;
		texel *= mix(uObjectColor, uObjectColor2, clamp((topatpoint - pixelpos.y) / (topatpoint - bottomatpoint), 0.0, 1.0));
	}

	return desaturate(texel);
}

// Check if light is in shadow according to its 1D shadow map
#ifdef SHADER_PASS_SHADOWMAPS
float shadowDirToU(vec2 dir) {
	if(abs(dir.y) > abs(dir.x)) {
		float x = dir.x / dir.y * 0.125;
		if(dir.y >= 0.0) {
			return 0.125 + x;
		}
		return 0.625 + x;
	}
	float y = dir.y / dir.x * 0.125;
	if(dir.x >= 0.0) {
		return 0.375 - y;
	}
	return 0.875 - y;
}

float sampleShadowmap(vec3 ray, float v) {
	float bias = 1.0;
	float negD = dot(vWorldNormal, ray);
	vec2 isize = textureSize(SHADER_PASS_SHADOWMAPS, 0);
	float scale = float(isize.x) * 0.25;

	// Snap to shadow map texel grid
	if(abs(ray.z) > abs(ray.x)) {
		ray.xy /= abs(ray.z);
		ray.x = (floor((ray.x + 1.0) * 0.5 * scale) + 0.5) / scale * 2.0 - 1.0;
		ray.z = sign(ray.z);
	} else {
		ray.yz /= abs(ray.x);
		ray.z = (floor((ray.z + 1.0) * 0.5 * scale) + 0.5) / scale * 2.0 - 1.0;
		ray.x = sign(ray.x);
	}

	float t = negD / dot(vWorldNormal, ray) - bias;
	vec2 dir = ray.xz * t;

	float u = shadowDirToU(dir);
	float dist2 = dot(dir, dir);
	return step(dist2, texture(SHADER_PASS_SHADOWMAPS, vec2(u, v)).x);
}

float shadowAttenuation(vec4 lightpos, float lightcolorA) {
	float shadowIndex = abs(lightcolorA);
	if(shadowIndex >= 16384.0) { return 1.0; } // No shadowmap available for this light. SHADOWMAP_INVALID_LIGHT

	// nudge light position slightly as maps tend to have their lights perfectly aligned with planes
	vec3 planePoint = pixelpos.xyz - lightpos.xyz + 0.01;
	if(dot(planePoint.xz, planePoint.xz) < 1.0) { return 1.0; } // Light is too close

	float v = (shadowIndex - 0.5) / MAX_SHADOWMAPPED_LIGHTS;
#ifdef SHADER_PASS_SMOOTH_SHADOWS
	// sigma male shadowmap interpolation
	#define planeSmoothSamples 2
	#define GAUSSIAN_RADIUS 2  // Expands kernel to 5x5
	#define GAUSSIAN_SIGMA_SQUARED 16.0  // Controls blur spread
	if(vWorldNormal.y != 0) {
		float sum = 0.0;
		float weightSum = 0.0;
		for(int x = -GAUSSIAN_RADIUS; x <= GAUSSIAN_RADIUS; x++) {
			for(int y = -GAUSSIAN_RADIUS; y <= GAUSSIAN_RADIUS; y++) {
				vec2 samplePos = pixelpos.xz + vec2(x, y) * (planeSmoothSamples / float(GAUSSIAN_RADIUS));
				float d = length(vec2(x, y));
				float weight = exp(-0.5 * (d * d) / GAUSSIAN_SIGMA_SQUARED);
				sum += sampleShadowmap(vec3(samplePos.x, pixelpos.y, samplePos.y) - lightpos.xyz + 0.01, v) * weight;
				weightSum += weight;
			}
		}
		return min(1.0, sum / weightSum);
	}
#endif
	return sampleShadowmap(planePoint, v);
}
#else
float shadowAttenuation(vec4 lightpos, float lightcolorA) {
	return 1.0;
}
#endif

// Adjust normal vector according to the normal map
#if defined(NORMALMAP)
mat3 cotangent_frame(vec3 n, vec3 p, vec2 uv) {
	// get edge vectors of the pixel triangle
	vec3 dp1 = dFdx(p);
	vec3 dp2 = dFdy(p);
	vec2 duv1 = dFdx(uv);
	vec2 duv2 = dFdy(uv);

	// solve the linear system
	vec3 dp2perp = cross(n, dp2); // cross(dp2, n);
	vec3 dp1perp = cross(dp1, n); // cross(n, dp1);
	vec3 t = dp2perp * duv1.x + dp1perp * duv2.x;
	vec3 b = dp2perp * duv1.y + dp1perp * duv2.y;

	// construct a scale-invariant frame
	float invmax = inversesqrt(max(dot(t, t), dot(b, b)));
	return mat3(t * invmax, b * invmax, n);
}

vec3 ApplyNormalMap(vec2 texcoord) {
#define WITH_NORMALMAP_UNSIGNED
#define WITH_NORMALMAP_GREEN_UP
	// #define WITH_NORMALMAP_2CHANNEL

	vec3 interpolatedNormal = normalize(vWorldNormal);

	vec3 map = texture(normaltexture, texcoord).xyz;
#if defined(WITH_NORMALMAP_UNSIGNED)
	map = map * 255. / 127. - 128. / 127.; // Math so "odd" because 0.5 cannot be precisely described in an unsigned format
#endif
#if defined(WITH_NORMALMAP_2CHANNEL)
	map.z = sqrt(1 - dot(map.xy, map.xy));
#endif
#if defined(WITH_NORMALMAP_GREEN_UP)
	map.y = -map.y;
#endif

	mat3 tbn = cotangent_frame(interpolatedNormal, pixelpos.xyz, vTexCoord);
	vec3 bumpedNormal = normalize(tbn * map);
	return bumpedNormal;
}
#else
vec3 ApplyNormalMap(vec2 texcoord) {
	return normalize(vWorldNormal);
}
#endif

// Main shader routine
void main() {
	Material material = ProcessMaterial();
#ifndef NO_ALPHATEST
	if(material.Base.a <= uAlphaThreshold) {
		discard;
	}
#endif

	// Normal pass, there's also a simpler, more direct pass when fog isn't enabled.
	if(uFogDensity != 0.0) {
		vec3 vertexColor = vColor.rgb;
		float fogdist = max(16.0, distance(pixelpos.xyz, uCameraPos.xyz));
		if(uFogDensity < 0.0) { vertexColor = mix(vec3(0.0, 0.0, 0.0), vertexColor, exp2(uFogDensity * fogdist)); } // black fog - used as a light level instead.
		// Handle glowing walls.
		if(uGlowTopColor.a > 0.0) {
			float glowTopPer = (uGlowTopPlane.w + uGlowTopPlane.x * pixelpos.x + uGlowTopPlane.y * pixelpos.z) * uGlowTopPlane.z - pixelpos.y;
			if(glowTopPer < uGlowTopColor.a) { vertexColor += (uGlowTopColor * (1.0 - glowTopPer / uGlowTopColor.a)).rgb; }
		}
		if(uGlowBottomColor.a > 0.0) {
			float glowBottomPer = pixelpos.y - (uGlowBottomPlane.w + uGlowBottomPlane.x * pixelpos.x + uGlowBottomPlane.y * pixelpos.z) * uGlowBottomPlane.z;
			if(glowBottomPer < uGlowBottomColor.a) { vertexColor += (uGlowBottomColor * (1.0 - glowBottomPer / uGlowBottomColor.a)).rgb; }
		}
		vec4 frag = vec4(ProcessMaterialLight(material, MaterialInitialLightValue(material, vertexColor)), material.Base.a * vColor.a);
		if(uFogDensity > 0.0) { frag.rgb = mix(uFogColor.rgb, frag.rgb, exp2(-uFogDensity * fogdist)); } // colored fog - affects at the end.
		FragColor = frag;
	} else {
		FragColor = vec4(ProcessMaterialLight(material, MaterialInitialLightValue(material, vColor.rgb)), material.Base.a * vColor.a);
	}

#ifdef SHADER_PASS_GBUFFER
	FragNormal = vec4(vEyeNormal * 0.5 + 0.5, 1.0);
#endif
}